document.addEventListener('DOMContentLoaded', function() {
    const fetchBtn = document.getElementById('fetch-btn');
    const videoUrlInput = document.getElementById('video-url');
    const resolutionOptions = document.getElementById('resolution-options');
    const resButtonsContainer = document.getElementById('res-buttons');
    const previewSection = document.getElementById('preview-section');
    const thumbnailPreview = document.getElementById('thumbnail-preview');
    
    fetchBtn.addEventListener('click', fetchThumbnails);
    videoUrlInput.addEventListener('keypress', function(e) {
        if (e.key === 'Enter') {
            fetchThumbnails();
        }
    });
    
    function fetchThumbnails() {
        const videoUrl = videoUrlInput.value.trim();
        
        if (!videoUrl) {
            alert('Please enter a YouTube video URL');
            return;
        }
        
        const videoId = extractVideoId(videoUrl);
        
        if (!videoId) {
            alert('Invalid YouTube URL. Please enter a valid YouTube video URL.');
            return;
        }
        
        // Clear previous options
        resButtonsContainer.innerHTML = '';
        
        // Define available resolutions
        const resolutions = [
            { name: 'Max Resolution', value: 'maxresdefault', size: '1280x720' },
            { name: 'High Quality', value: 'hqdefault', size: '480x360' },
            { name: 'Medium Quality', value: 'mqdefault', size: '320x180' },
            { name: 'Standard Quality', value: 'sddefault', size: '640x480' },
            { name: 'Default', value: 'default', size: '120x90' }
        ];
        
        // Create buttons for each resolution
        resolutions.forEach(res => {
            const btn = document.createElement('button');
            btn.className = 'res-btn';
            btn.textContent = `${res.name} (${res.size})`;
            btn.dataset.res = res.value;
            btn.addEventListener('click', function() {
                showThumbnail(videoId, res.value, res.size);
            });
            resButtonsContainer.appendChild(btn);
        });
        
        // Show resolution options
        resolutionOptions.classList.remove('hidden');
        
        // Scroll to resolution options
        resolutionOptions.scrollIntoView({ behavior: 'smooth' });
    }
    
    function extractVideoId(url) {
        // Regular expressions for different YouTube URL formats
        const regExp = /^.*(youtu.be\/|v\/|u\/\w\/|embed\/|watch\?v=|&v=)([^#&?]*).*/;
        const match = url.match(regExp);
        
        return (match && match[2].length === 11) ? match[2] : null;
    }
    
    function showThumbnail(videoId, resolution, size) {
        const thumbnailUrl = `https://img.youtube.com/vi/${videoId}/${resolution}.jpg`;
        
        // Set preview image source
        thumbnailPreview.src = thumbnailUrl;
        thumbnailPreview.alt = `YouTube thumbnail - ${size}`;
        
        // Open the image in a new tab when clicked
        thumbnailPreview.onclick = function() {
            window.open(thumbnailUrl, '_blank');
        };
        
        // Show preview section
        previewSection.classList.remove('hidden');
        
        // Scroll to preview section
        previewSection.scrollIntoView({ behavior: 'smooth' });
        
        // Add download functionality
        setTimeout(() => {
            thumbnailPreview.setAttribute('title', 'Click to view full size. Right click → Save image as to download.');
        }, 500);
    }
    
    // Smooth scrolling for anchor links
    document.querySelectorAll('a[href^="#"]').forEach(anchor => {
        anchor.addEventListener('click', function(e) {
            e.preventDefault();
            
            const targetId = this.getAttribute('href');
            if (targetId === '#') return;
            
            const targetElement = document.querySelector(targetId);
            if (targetElement) {
                targetElement.scrollIntoView({
                    behavior: 'smooth'
                });
            }
        });
    });
});